<?php
session_start(); // 세션 시작 (로그인 정보, 찜 목록 등 사용자 상태 유지에 필요)
include('db.php'); // DB 연결 파일 포함 (MySQL 연결 객체 $conn 사용 가능)

// ------------------------------
// 찜한 책 목록 가져오기
// ------------------------------
// 찜 목록을 세션에서 가져온다 (없으면 빈 배열)
$favorites = $_SESSION['favorites'] ?? [];
$books = []; // 찜한 책 정보들을 저장할 배열 초기화

// ------------------------------
// 찜 목록에서 책 삭제 요청 처리
// ------------------------------
// POST 방식으로 delete_id가 넘어오면 찜 목록에서 해당 책 ID를 삭제한다
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_id'])) {
    $delete_id = intval($_POST['delete_id']); // 삭제할 책 ID 정수 변환
    // 찜 목록에서 삭제할 ID가 있는지 찾기
    if (($key = array_search($delete_id, $favorites)) !== false) {
        unset($favorites[$key]); // 찜 목록에서 해당 요소 제거
        $_SESSION['favorites'] = array_values($favorites); // 인덱스 재정렬 후 다시 세션에 저장
        header("Location: wishlist.php"); // 페이지 새로고침 (삭제 후 새로고침해서 반영)
        exit; // 종료
    }
}

// ------------------------------
// 찜한 책이 있으면 DB에서 책 정보 가져오기
// ------------------------------
if (!empty($favorites)) {
    // 찜 목록 개수만큼 ? 자리 만들어서 prepare문 구성 (SQL Injection 방지용)
    $placeholders = implode(',', array_fill(0, count($favorites), '?'));
    $types = str_repeat('i', count($favorites)); // 책 ID는 정수형이므로 'i' 반복

    // prepare문 생성 (찜 목록에 있는 책들을 모두 가져오기)
    $stmt = $conn->prepare("SELECT * FROM books WHERE id IN ($placeholders)");
    $stmt->bind_param($types, ...$favorites); // 가변 인자 방식으로 바인딩
    $stmt->execute(); // 쿼리 실행
    $result = $stmt->get_result(); // 결과 저장

    // 결과를 $books 배열에 저장
    while ($row = $result->fetch_assoc()) {
        $books[] = $row;
    }
    $stmt->close(); // prepare문 닫기
}
?>

<!DOCTYPE html>
<html lang="ko">

<head>
    <meta charset="UTF-8">
    <title>보관함 - ReadUS</title>
    <link rel="stylesheet" href="css/styles.css">
</head>
<style>
    body {
        margin: 0 auto;
        font-family: Arial, sans-serif;
        background-color: #f3f4f6;
        width: 1280px;
    }
</style>
<body>
    <!-- 헤더 섹션: 상단 로고 및 메뉴 -->
    <div class="header page-section">
        <div class="logo">
            <a href="index.php">ReadUS</a> <!-- 클릭 시 홈으로 이동 -->
        </div>
        <div class="menu">
            <?php if (isset($_SESSION['loggedin']) && $_SESSION['loggedin'] === true): ?>
                <!-- 로그인 되어있으면 사용자 이름과 주요 메뉴 링크 보여주기 -->
                <span class="welcome-message"><?php echo htmlspecialchars($_SESSION['username']); ?>님</span>
                <a href="edit-profile.php" title="회원정보 수정">⚙️</a>
                <a href="logout.php" title="로그아웃">🚪</a>
                <a href="mypage.php" title="MyPage">📚</a>
                <a href="wishlist.php" title="보관함">💖</a>
                <a href="cart.php" title="장바구니">🛒</a>
            <?php else: ?>
                <!-- 로그인 안 되어있으면 로그인, 회원가입 링크만 보여주기 -->
                <a href="login.php" title="로그인">🔐</a>
                <a href="signup.php" title="회원가입">👤</a>
            <?php endif; ?>
        </div>
    </div>

    <main>
        <h2>💖 보관함</h2>
        <!-- 찜한 책이 없으면 메시지 출력 -->
        <?php if (empty($books)): ?>
            <p>찜한 도서가 없습니다.</p>
        <?php else: ?>
            <!-- 찜한 책이 있으면 각각 책 정보 출력 -->
            <?php foreach ($books as $book): ?>
                <div class="book-item">
                    <div class="book-image">
                        <!-- 책 표지 이미지 출력, 없으면 기본 이미지 사용 -->
                        <img src="img/<?php echo htmlspecialchars($book['cover_image'] ?: 'default-book-cover.jpg'); ?>" alt="책 표지">
                    </div>
                    <div class="book-info">
                        <div class="book-title-wrapper">
                            <!-- 책 제목 -->
                            <div class="book-title"><?php echo htmlspecialchars($book['title']); ?></div>
                            <!-- 삭제 폼: 버튼 클릭시 해당 책 ID를 POST로 보내 삭제 처리 -->
                            <form method="post" onsubmit="return confirm('이 책을 보관함에서 삭제하시겠습니까?');">
                                <input type="hidden" name="delete_id" value="<?php echo $book['id']; ?>">
                                <button type="submit" class="delete-button">❌</button>
                            </form>
                        </div>
                        <!-- 책 상세 정보들 -->
                        <div class="book-author">저자: <?php echo htmlspecialchars($book['author']); ?></div>
                        <div class="book-publisher">출판사: <?php echo htmlspecialchars($book['publisher']); ?></div>
                        <div class="book-date">출간일: <?php echo htmlspecialchars($book['published_date']); ?></div>
                        <!-- 줄바꿈 유지하며 설명 표시 -->
                        <div class="book-description"><?php echo nl2br(htmlspecialchars($book['description'])); ?></div>
                        <div class="book-price">가격: <?php echo number_format($book['price']); ?> 원</div>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </main>

    <!-- 푸터 섹션: 연락처 등 정보 -->
    <div class="footer-box page-section">
        <div class="footer-item">안산대학교</div>
        <div class="footer-item">010-0000-0000</div>
        <div class="footer-item">shinyourim@ansan.ac.kr</div>
    </div>
</body>

</html>
