<?php
// 세션 시작 (로그인 상태 확인 및 사용자 정보 유지)
session_start();

// DB 연결 파일 불러오기
include('db.php');

// 로그인 여부 확인
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    // 로그인 안 되어 있으면 로그인 페이지로 이동
    header('Location: login.php');
    exit;
}

// 로그인된 사용자 정보 저장
$userId = $_SESSION['user_id'];
$username = $_SESSION['username'];

// 메시지 초기화
$successMessage = '';
$errorMessage = '';

// -----------------------------
// 친구 목록 불러오기
// -----------------------------
$friends = [];
$sql = "
    SELECT u.id, u.username
    FROM users u
    INNER JOIN friends f ON u.id = f.friend_id
    WHERE f.user_id = ?
";
$stmt = $conn->prepare($sql); // SQL 준비
$stmt->bind_param('i', $userId); // 사용자 ID 바인딩
$stmt->execute(); // 쿼리 실행
$result = $stmt->get_result(); // 결과 받기
while ($row = $result->fetch_assoc()) {
    $friends[] = $row; // 배열에 친구 추가
}
$stmt->close(); // 메모리 해제

// -----------------------------
// 책 목록 불러오기
// -----------------------------
$books = [];
$sqlBooks = "SELECT id, title FROM books ORDER BY title ASC"; // 제목 기준 정렬
$resultBooks = $conn->query($sqlBooks);
if ($resultBooks) {
    while ($rowBook = $resultBooks->fetch_assoc()) {
        $books[] = $rowBook; // 책 배열에 추가
    }
}

// -----------------------------
// 책 추천 POST 처리
// -----------------------------
if ($_SERVER["REQUEST_METHOD"] === "POST") {
    // 폼에서 입력받은 데이터
    $friendId = $_POST['friend_id'];
    $bookId = $_POST['book_id'];
    $bookComment = trim($_POST['book_comment']); // 메시지 공백 제거

    // 필수값 검사
    if ($friendId && $bookId) {
        $created_at = date('Y-m-d H:i:s'); // 현재 날짜/시간

        // 추천 정보를 DB에 저장
        $sql = "INSERT INTO recommendations (receiver_id, book_id, message, created_at, from_user_id) VALUES (?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            // 쿼리 준비 오류
            $errorMessage = "쿼리 준비 실패: " . $conn->error;
        } else {
            $stmt->bind_param("iissi", $friendId, $bookId, $bookComment, $created_at, $userId);
            if ($stmt->execute()) {
                $successMessage = "책 추천이 완료되었습니다!";
            } else {
                $errorMessage = "추천 저장 중 오류가 발생했습니다.";
            }
            $stmt->close();
        }
    } else {
        $errorMessage = "모든 필드를 입력해주세요.";
    }
}
?>

<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <title>책 추천하기 - ReadUS</title>
    <link rel="stylesheet" href="css/styles.css">
    <style>
        /* 페이지 전체 배경 및 여백 설정 */
        #recommend-page {
            background-color: #fafafa;
            padding: 0 15px 30px;
        }

        /* 전체 컨테이너 중앙 정렬 */
        .recommend-container {
            max-width: 1280px;
            margin: 0 auto;
        }

        /* 상단 메뉴 설정 */
        .recommend-menu {
            display: flex;
            align-items: center;
        }

        /* 메뉴 내 링크 디자인 */
        .recommend-menu a {
            text-decoration: none;
            color: #333;
            font-size: 28px;
            margin-left: 30px;
            transition: transform 0.2s, color 0.3s;
            position: relative;
        }

        /* 메뉴 hover 시 효과 */
        .recommend-menu a:hover {
            color: #a499f2;
            transform: scale(1.2);
        }

        /* 메뉴 hover 시 말풍선 툴팁 */
        .recommend-menu a::after {
            content: attr(title);
            position: absolute;
            bottom: -35px;
            left: 50%;
            transform: translateX(-50%);
            background-color: #333;
            color: #fff;
            padding: 6px 10px;
            border-radius: 6px;
            font-size: 12px;
            opacity: 0;
            transition: opacity 0.3s, transform 0.3s;
            z-index: 10;
        }

        .recommend-menu a:hover::after {
            opacity: 1;
            transform: translateX(-50%) translateY(5px);
        }

        /* 로그인 사용자 환영 메시지 */
        .welcome-message {
            background-color: #e6e6fa;
            border-radius: 16px;
            font-weight: 600;
            font-size: 14px;
            padding: 6px 24px;
            margin: 0 auto;
            display: flex;
            justify-content: center;
            align-items: center;
        }

        /* 추천 폼 박스 */
        .recommend-form-wrapper {
            max-width: 600px;
            margin: 0 auto;
            background-color: #fff;
            padding: 25px 30px;
            border-radius: 12px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }

        .recommend-form-wrapper label {
            margin: 20px 0 8px;
            display: block;
            font-weight: 600;
        }

        .recommend-form-wrapper select,
        .recommend-form-wrapper textarea {
            width: 100%;
            padding: 10px 15px;
            border: 1.5px solid #ccc;
            border-radius: 8px;
        }

        .recommend-form-wrapper textarea {
            resize: vertical;
            min-height: 100px;
        }

        /* 추천 버튼 스타일 */
        .recommend-form-wrapper button {
            margin-top: 30px;
            width: 100%;
            background-color: #a499f2;
            color: #fff;
            font-weight: 700;
            padding: 14px 0;
            font-size: 1.2rem;
            border-radius: 10px;
            cursor: pointer;
        }

        .recommend-form-wrapper button:hover {
            background-color: #e6e6fa;
        }

        /* 성공 메시지 (우측 하단 고정) */
        p.success-message {
            position: fixed;
            bottom: 20px;
            right: 20px;
            background-color: #000;
            color: #fff;
            font-weight: 700;
            padding: 12px 20px;
            border-radius: 10px;
            z-index: 1000;
        }

        /* 에러 메시지 */
        p.error-message {
            background-color: #f8d7da;
            color: #721c24;
            font-weight: 700;
            padding: 12px 20px;
            border-radius: 10px;
            text-align: center;
            max-width: 600px;
            margin: 20px auto 0;
        }

        /* 친구 없을 때 안내문 */
        p.no-friends {
            font-style: italic;
            text-align: center;
            color: #888;
            max-width: 600px;
            margin: 20px auto 0;
        }

        p.no-friends a {
            font-weight: 600;
            color: #0052cc;
            text-decoration: none;
        }

        p.no-friends a:hover {
            text-decoration: underline;
        }
    </style>
</head>

<body id="recommend-page">
    <div class="recommend-container">
        <!-- 헤더 영역 -->
        <header class="header">
            <div class="logo">
                <a href="index.php">ReadUS</a>
            </div>
            <nav class="recommend-menu">
                <?php if (isset($_SESSION['loggedin']) && $_SESSION['loggedin'] === true): ?>
                    <!-- 로그인 상태라면 메뉴 표시 -->
                    <span class="welcome-message"><?php echo htmlspecialchars($username); ?>님</span>
                    <a href="wishlist.php" title="보관함">💖</a>
                    <a href="cart.php" title="장바구니">🛒</a>
                    <a href="mypage.php" title="MyPage">📚</a>
                    <a href="edit-profile.php" title="회원정보 수정">⚙️</a>
                    <a href="logout.php" title="로그아웃">🚪</a>
                <?php else: ?>
                    <!-- 비로그인 시 로그인/회원가입 -->
                    <a href="login.php" title="로그인">🔐</a>
                    <a href="signup.php" title="회원가입">👤</a>
                <?php endif; ?>
            </nav>
        </header>

        <!-- 본문 -->
        <div class="page-section">
            <h2 style="text-align:center;">📚 책 추천하기</h2>
        </div>

        <!-- 추천 완료 또는 에러 메시지 출력 -->
        <?php if ($successMessage): ?>
            <p class="success-message"><?php echo $successMessage; ?></p>
        <?php elseif ($errorMessage): ?>
            <p class="error-message"><?php echo $errorMessage; ?></p>
        <?php endif; ?>

        <!-- 친구가 없을 경우 -->
        <?php if (empty($friends)): ?>
            <p class="no-friends">추천할 친구가 없습니다. <a href="friends.php">친구 추가하기</a></p>
        <?php else: ?>
            <!-- 책 추천 폼 -->
            <form class="recommend-form-wrapper" method="POST" action="recommend-book.php">
                <label for="friend_id">추천할 친구:</label>
                <select id="friend_id" name="friend_id" required>
                    <option value="">-- 선택하세요 --</option>
                    <?php foreach ($friends as $friend): ?>
                        <option value="<?php echo $friend['id']; ?>" <?php echo (isset($_POST['friend_id']) && $_POST['friend_id'] == $friend['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($friend['username']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>

                <label for="book_id">책 선택:</label>
                <select id="book_id" name="book_id" required>
                    <option value="">-- 선택하세요 --</option>
                    <?php foreach ($books as $book): ?>
                        <option value="<?php echo $book['id']; ?>" <?php echo (isset($_POST['book_id']) && $_POST['book_id'] == $book['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($book['title']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>

                <label for="book_comment">한마디 (선택):</label>
                <textarea id="book_comment" name="book_comment" rows="4" cols="40" placeholder="간단한 한마디를 남겨보세요 (선택)">
                    <?php echo isset($_POST['book_comment']) ? htmlspecialchars($_POST['book_comment']) : ''; ?>
                </textarea>

                <button type="submit">책 추천하기</button>
            </form>
        <?php endif; ?>
    </div>
</body>
</html>
