<?php
session_start();  // 세션 시작

// 로그인 상태가 아니면 로그인 페이지로 이동
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
  header("Location: login.php");
  exit;
}

require_once 'db.php';  // DB 연결 파일 포함

// 세션에서 사용자 ID 가져오기
$user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;

// 세션에 사용자 ID가 없으면 에러 메시지 출력 후 종료
if ($user_id === null) {
  die("❌ 세션이 유효하지 않습니다.");
}

$error_message = "";    // 에러 메시지 저장용 변수 초기화
$success_message = "";  // 성공 메시지 저장용 변수 초기화

// 사용자 정보 불러오기 (username, email)
$sql = "SELECT username, email FROM users WHERE id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$stmt->close();

// POST 요청 처리 (폼 제출 시)
if ($_SERVER["REQUEST_METHOD"] == "POST") {
  if (isset($_POST['update_profile'])) {
    // 회원 정보 수정 처리
    $new_email = $_POST['email'];                // 새 이메일
    $new_password = $_POST['password'];          // 새 비밀번호
    $confirm_password = $_POST['confirm_password'];  // 비밀번호 확인

    // 비밀번호와 확인란이 다르면 에러 메시지 저장
    if (!empty($new_password) && $new_password !== $confirm_password) {
      $error_message = "❌ 비밀번호가 일치하지 않습니다.";
    } else {
      if (!empty($new_password)) {
        // 비밀번호가 비어있지 않으면 비밀번호도 같이 업데이트
        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT); // 비밀번호 해시 생성
        $sql = "UPDATE users SET email = ?, password = ? WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ssi", $new_email, $hashed_password, $user_id);
      } else {
        // 비밀번호 변경하지 않고 이메일만 업데이트
        $sql = "UPDATE users SET email = ? WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("si", $new_email, $user_id);
      }

      // 쿼리 실행 성공 시 성공 메시지 저장 및 세션 이메일 업데이트
      if ($stmt->execute()) {
        $success_message = "✅ 회원정보가 성공적으로 수정되었습니다!";
        $_SESSION['email'] = $new_email;
      } else {
        // 쿼리 실행 실패 시 에러 메시지 저장
        $error_message = "❌ 수정에 실패했습니다. 다시 시도해주세요.";
      }
      $stmt->close();
    }
  } elseif (isset($_POST['delete_account'])) {
    // 회원 탈퇴 처리

    // user_read_books 테이블에서 해당 사용자의 데이터 삭제
    $stmt = $conn->prepare("DELETE FROM user_read_books WHERE user_id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $stmt->close();

    // friends 테이블에서 해당 사용자가 포함된 친구 관계 삭제 (양방향 모두)
    $stmt = $conn->prepare("DELETE FROM friends WHERE user_id = ? OR friend_id = ?");
    $stmt->bind_param("ii", $user_id, $user_id);
    $stmt->execute();
    $stmt->close();

    // users 테이블에서 사용자 정보 삭제
    $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
    $stmt->bind_param("i", $user_id);

    if ($stmt->execute()) {
      session_destroy();               // 세션 종료 (로그아웃 처리)
      header("Location: goodbye.php"); // 탈퇴 후 인사 페이지로 이동
      exit;
    } else {
      $error_message = "❌ 탈퇴에 실패했습니다. 다시 시도해주세요.";
    }

    $stmt->close();
  }
}

$conn->close(); // DB 연결 종료
?>

<!DOCTYPE html>
<html lang="ko">

<head>
  <meta charset="UTF-8">
  <title>회원정보 수정</title>
  <link rel="stylesheet" href="css/styles.css">
  <style>
    body {
      margin: 0;
      font-family: Arial, sans-serif;
      background-color: #f3f4f6;
    }

    .container {
      width: 1280px;
      margin: 0 auto; /* 가운데 정렬 */
      background-color: white; /* 필요시 배경색 */
    }

    .delete-btn {
      background-color: #e74c3c; /* 빨간색 배경 */
      color: white;              /* 흰색 글씨 */
      margin-top: 10px;
    }
  </style>
</head>

<body class="signup-page">
  <div class="auth-container">
    <div class="auth-logo"><a href="index.php">ReadUS</a></div>
    <div class="auth-header">
      <h2>회원정보 수정</h2>
    </div>

    <!-- 에러 메시지 출력 -->
    <?php if ($error_message): ?>
      <div class="error-message">
        <p><?= htmlspecialchars($error_message) ?></p>
      </div>
    <?php endif; ?>

    <!-- 성공 메시지 출력 -->
    <?php if ($success_message): ?>
      <div class="success-message">
        <p><?= htmlspecialchars($success_message) ?></p>
      </div>
    <?php endif; ?>

    <form method="POST" action="edit-profile.php">
      <div class="input-box">
        <label for="username">아이디</label>
        <input type="text" id="username" name="username" value="<?= htmlspecialchars($user['username']) ?>" readonly>
        <!-- 아이디는 수정 불가 -->
      </div>

      <div class="input-box">
        <label for="email">이메일</label>
        <input type="email" id="email" name="email" value="<?= htmlspecialchars($user['email']) ?>" required>
        <!-- 이메일 수정 가능 -->
      </div>

      <div class="input-box">
        <label for="password">새 비밀번호 (선택)</label>
        <input type="password" id="password" name="password" placeholder="새 비밀번호 입력">
        <!-- 새 비밀번호 입력란 -->
      </div>

      <div class="input-box">
        <label for="confirm_password">비밀번호 확인</label>
        <input type="password" id="confirm_password" name="confirm_password" placeholder="비밀번호 다시 입력">
        <!-- 비밀번호 확인 입력란 -->
      </div>

      <button class="auth-btn" type="submit" name="update_profile">수정 완료</button>
      <!-- 회원정보 수정 제출 버튼 -->

      <button class="auth-btn delete-btn" type="submit" name="delete_account"
        onclick="return confirm('정말로 회원 탈퇴하시겠습니까? 이 작업은 되돌릴 수 없습니다.');">
        회원 탈퇴
      </button>
      <!-- 회원 탈퇴 버튼, 클릭 시 확인창 표시 -->

      <div class="auth-footer"><a href="index.php">ReadUS</a></div>
    </form>
  </div>
</body>

</html>
