<?php
session_start();
include('db.php'); // 데이터베이스 연결 파일 포함

// 로그인 여부 확인, 로그인 안 되어 있으면 로그인 페이지로 이동
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    header('Location: login.php');
    exit();
}

// 세션에 저장된 장바구니 불러오기, 비어있으면 장바구니 페이지로 이동
$cart = $_SESSION['cart'] ?? [];
if (empty($cart)) {
    $_SESSION['flash_message'] = "장바구니가 비어있습니다.";
    header('Location: cart.php');
    exit();
}

$booksInCart = []; // 장바구니에 담긴 도서 정보 저장할 배열
$totalPrice = 0;   // 총 합계 금액 초기화

// 장바구니에 담긴 도서 id 갯수만큼 물음표(?) 플레이스홀더 생성 (SQL 인젝션 방지)
$placeholders = implode(',', array_fill(0, count($cart), '?'));
$types = str_repeat('i', count($cart)); // 바인딩 타입 지정 (모두 정수형 'i')
$stmt = $conn->prepare("SELECT * FROM books WHERE id IN ($placeholders)"); // 도서 정보 조회 준비

$ids = array_keys($cart); // 장바구니에 담긴 도서 id들
$params = [];
$params[] = &$types;      // 첫번째 파라미터는 타입 문자열
foreach ($ids as $key => $id) {
    $params[] = &$ids[$key];  // 나머지 파라미터는 id 값들
}

// 동적으로 파라미터 바인딩 (bind_param은 가변인자 사용)
call_user_func_array([$stmt, 'bind_param'], $params);

$stmt->execute();
$result = $stmt->get_result();

// 조회된 도서 정보를 배열에 저장, 수량과 합계 계산
while ($row = $result->fetch_assoc()) {
    $row['quantity'] = $cart[$row['id']]; // 장바구니 수량
    $row['subtotal'] = $row['price'] * $row['quantity']; // 개별 도서 합계
    $totalPrice += $row['subtotal']; // 전체 합계 누적
    $booksInCart[] = $row;
}
$stmt->close();

$error = "";       // 오류 메시지 초기화
$kakaoQRUrl = '';  // 카카오페이 QR코드 URL 초기화

// 결제 폼 제출 시 처리
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['pay'])) {
    // 입력값 가져오기 및 공백 제거
    $name = trim($_POST['name']);
    $phone = trim($_POST['phone']);
    $address = trim($_POST['address']);
    $payment_method = $_POST['payment_method'];

    // 필수 정보 누락 시 오류 메시지 설정
    if (empty($name) || empty($phone) || empty($address)) {
        $error = "모든 정보를 입력해주세요.";
    } else {
        if ($payment_method === 'kakao') {
            // TODO: 실제 카카오페이 API 연동 필요
            // 현재는 테스트용 임시 QR코드 이미지 URL 사용
            $kakaoQRUrl = "https://yourdomain.com/kakaopay_qr_sample.png";
        } else {
            // 신용카드나 무통장입금 등 일반 결제 처리

            // 주문 번호 생성 (날짜 + 랜덤 숫자)
            $orderId = date('YmdHis') . rand(100, 999);
            $paymentDate = date('Y-m-d H:i:s');

            $items = $booksInCart;
            $totalPriceCalc = 0;
            foreach ($items as &$item) {
                $item['quantity'] = $cart[$item['id']];
                $item['subtotal'] = $item['price'] * $item['quantity'];
                $totalPriceCalc += $item['subtotal'];
            }

            // 주문 정보를 세션에 저장
            $_SESSION['last_order'] = [
                'order_id' => $orderId,
                'payment_date' => $paymentDate,
                'items' => $items,
                'total_price' => $totalPriceCalc,
                'customer_name' => $name,
                'customer_phone' => $phone,
                'customer_address' => $address,
                'payment_method' => $payment_method,
            ];

            // 결제 후 장바구니 비우기
            $_SESSION['cart'] = [];
            $_SESSION['flash_message'] = "결제가 완료되었습니다. 감사합니다!";
            header('Location: order_complete.php'); // 주문 완료 페이지로 이동
            exit();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="ko">

<head>
    <meta charset="UTF-8">
    <title>결제하기 - ReadUS</title>
    <link rel="stylesheet" href="css/styles.css">
    <style>
        /* 본문, 레이아웃, 폼, 테이블 등 스타일 정의 */
        body {
            margin: 0 auto;
            font-family: Arial, sans-serif;
            background-color: #f3f4f6;
            width: 1280px;
        }

        main {
            max-width: 720px;
            margin: 40px auto;
            background: white;
            padding: 30px 40px;
            border-radius: 12px;
            box-shadow: 0 10px 25px rgba(42, 114, 217, 0.15);
        }

        h2,
        h3 {
            color: #000000;
            margin-bottom: 20px;
            font-weight: 700;
        }

        h3 {
            margin-top: 40px;
        }

        .error-message {
            background: #ffebee;
            color: #c62828;
            padding: 12px 20px;
            border-radius: 8px;
            margin-bottom: 25px;
            font-weight: 600;
            border: 1px solid #f44336;
        }

        .cart-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0 10px;
            box-shadow: 0 5px 10px rgba(0, 0, 0, 0.05);
        }

        .cart-table thead th {
            background-color: #a499f2;
            color: white;
            padding: 12px 15px;
            text-align: left;
            font-weight: 700;
            border-radius: 10px 10px 0 0;
        }

        .cart-table tbody tr {
            background-color: #f9fbff;
            border-radius: 10px;
        }

        .cart-table tbody td {
            padding: 15px;
            vertical-align: middle;
            color: #333;
            font-size: 1rem;
        }

        .cart-table tfoot td {
            padding: 15px;
            font-weight: 700;
            font-size: 1.1rem;
            text-align: right;
            color: #222;
        }

        form label {
            display: block;
            font-weight: 600;
            margin-bottom: 6px;
            color: #444;
            font-size: 1.05rem;
        }

        form input[type="text"],
        form input[type="tel"],
        form select {
            width: 100%;
            padding: 12px 15px;
            font-size: 1rem;
            border: 1.8px solid #cbd4db;
            border-radius: 8px;
            margin-bottom: 25px;
            transition: border-color 0.3s ease;
        }

        form input[type="text"]:focus,
        form input[type="tel"]:focus,
        form select:focus {
            border-color: #a499f2;
            outline: none;
        }

        button[type="submit"] {
            background-color: #a499f2;
            color: white;
            padding: 15px 0;
            width: 100%;
            font-size: 1.15rem;
            font-weight: 700;
            border: none;
            border-radius: 10px;
            cursor: pointer;
            transition: background-color 0.3s ease;
            margin-top: 10px;
        }

        button[type="submit"]:hover {
            background-color: #e6e6fa;
        }

        .btn {
            display: inline-block;
            background-color: #a499f2;
            color: #ffffff;
            padding: 10px 25px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 600;
            margin-right: 15px;
            transition: background-color 0.3s ease;
        }

        .btn:hover {
            background-color: #e6e6fa;
        }

        .footer-item {
            font-size: 1rem;
        }

        @media (max-width: 600px) {
            main {
                padding: 20px;
                margin: 20px;
            }

            .footer-box.page-section {
                flex-direction: column;
                gap: 10px;
                font-size: 0.9rem;
            }
        }

        /* 카카오페이 QR 코드 영역 스타일 */
        .kakao-qr {
            margin-top: 20px;
            padding: 20px;
            background: #ffeb3b33;
            border-radius: 12px;
            text-align: center;
        }

        .kakao-qr img {
            width: 200px;
            height: 200px;
            border-radius: 8px;
        }
    </style>
</head>


<body>
    <div class="header page-section">
        <div class="logo">
            <a href="index.php">ReadUS</a> <!-- 로고 및 홈페이지 링크 -->
        </div>
        <div class="menu">
            <?php if (isset($_SESSION['loggedin']) && $_SESSION['loggedin'] === true): ?>
                <!-- 로그인 상태면 사용자명, 프로필 수정, 로그아웃, 마이페이지, 보관함, 장바구니 메뉴 -->
                <span class="welcome-message"><?php echo htmlspecialchars($_SESSION['username']); ?>님</span>
                <a href="edit-profile.php" title="회원정보 수정">⚙️</a>
                <a href="logout.php" title="로그아웃">🚪</a>
                <a href="mypage.php" title="MyPage">📚</a>
                <a href="wishlist.php" title="보관함">💖</a>
                <a href="cart.php" title="장바구니">🛒</a>
            <?php else: ?>
                <!-- 로그인 안 되어 있으면 로그인, 회원가입 링크 -->
                <a href="login.php" title="로그인">🔐</a>
                <a href="signup.php" title="회원가입">👤</a>
            <?php endif; ?>
        </div>
    </div>

    <main>
        <h2>결제하기</h2>

        <!-- 오류 메시지 출력 -->
        <?php if (!empty($error)): ?>
            <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>

        <h3>주문 내역</h3>
        <table class="cart-table">
            <thead>
                <tr>
                    <th>도서명</th>
                    <th>수량</th>
                    <th>가격</th>
                    <th>합계</th>
                </tr>
            </thead>
            <tbody>
                <!-- 장바구니에 담긴 도서 목록 반복 출력 -->
                <?php foreach ($booksInCart as $book): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($book['title']); ?></td>
                        <td><?php echo $book['quantity']; ?></td>
                        <td><?php echo number_format($book['price']); ?> 원</td>
                        <td><?php echo number_format($book['subtotal']); ?> 원</td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
            <tfoot>
                <tr>
                    <td colspan="3">총 합계</td>
                    <td><?php echo number_format($totalPrice); ?> 원</td>
                </tr>
            </tfoot>
        </table>

        <h3>배송 및 결제 정보 입력</h3>
        <form method="POST" action="checkout.php">
            <!-- 이름 입력 -->
            <label for="name">이름</label>
            <input type="text" name="name" id="name" required
                value="<?php echo isset($_POST['name']) ? htmlspecialchars($_POST['name']) : ''; ?>">

            <!-- 연락처 입력 -->
            <label for="phone">연락처</label>
            <input type="tel" name="phone" id="phone" required pattern="[0-9\-]+" placeholder="010-1234-5678"
                value="<?php echo isset($_POST['phone']) ? htmlspecialchars($_POST['phone']) : ''; ?>">

            <!-- 배송 주소 입력 -->
            <label for="address">배송 주소</label>
            <input type="text" name="address" id="address" required
                value="<?php echo isset($_POST['address']) ? htmlspecialchars($_POST['address']) : ''; ?>">

            <!-- 결제 수단 선택 -->
            <label for="payment_method">결제 수단</label>
            <select name="payment_method" id="payment_method" required>
                <option value="">선택하세요</option>
                <option value="card" <?php echo (isset($_POST['payment_method']) && $_POST['payment_method'] === 'card') ? 'selected' : ''; ?>>신용카드</option>
                <option value="bank" <?php echo (isset($_POST['payment_method']) && $_POST['payment_method'] === 'bank') ? 'selected' : ''; ?>>무통장입금</option>
            </select>

            <!-- 결제 버튼 -->
            <button type="submit" name="pay">결제하기</button>
        </form>

    <footer class="footer-box page-section">
        <div class="footer-item">shinyourim@ansan.ac.kr</div>
        <div class="footer-item">010-0000-0000</div>
    </footer>
</body>

</html>