<?php
session_start();  // 세션 시작
include('db.php');  // 데이터베이스 연결 파일 포함

// GET 파라미터에서 카테고리 이름과 검색어를 가져옴, 없으면 빈 문자열
$categoryName = $_GET['name'] ?? '';
$searchQuery = $_GET['search'] ?? '';
$books = [];  // 도서 목록 저장할 배열 초기화
$categoryId = null;  // 카테고리 ID 변수 초기화

// 카테고리 이름이 있을 경우, 해당 이름으로 카테고리 ID를 DB에서 조회
if ($categoryName) {
    $stmt = $conn->prepare("SELECT id FROM categories WHERE name = ?");
    $stmt->bind_param("s", $categoryName);
    $stmt->execute();
    $stmt->bind_result($categoryId);
    $stmt->fetch();
    $stmt->close();
}

// 기본 도서 조회 쿼리 (1=1은 조건절을 편하게 이어가기 위함)
$query = "SELECT * FROM books WHERE 1=1";
$params = [];  // 바인딩할 파라미터 배열
$types = "";   // 파라미터 타입 문자열

// 카테고리 ID가 있을 경우 해당 카테고리 도서만 조회하도록 조건 추가
if ($categoryId) {
    $query .= " AND category_id = ?";
    $params[] = $categoryId;
    $types .= "i";  // 정수형
}

// 검색어가 있을 경우 제목에 검색어가 포함된 도서 조회 조건 추가
if (!empty($searchQuery)) {
    $query .= " AND title LIKE ?";
    $params[] = "%" . $searchQuery . "%";  // 부분 일치 검색
    $types .= "s";  // 문자열
}

// 준비된 쿼리 생성
$stmt = $conn->prepare($query);

// 파라미터가 있으면 바인딩
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}

// 쿼리 실행 및 결과 저장
$stmt->execute();
$result = $stmt->get_result();

// 결과를 배열에 담기
while ($row = $result->fetch_assoc()) {
    $books[] = $row;
}
$stmt->close();

// POST 요청 시 찜하기 또는 장바구니 처리
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $bookId = (int) $_POST['book_id'];  // 선택된 책 ID

    // 해당 도서 제목을 DB에서 조회
    $stmt = $conn->prepare("SELECT title FROM books WHERE id = ?");
    $stmt->bind_param("i", $bookId);
    $stmt->execute();
    $stmt->bind_result($bookTitle);
    $stmt->fetch();
    $stmt->close();

    // 찜하기 버튼 클릭 시 세션의 favorites 배열에 도서 ID 추가
    if (isset($_POST['add_to_favorites'])) {
        if (!in_array($bookId, $_SESSION['favorites'] ?? [])) {
            $_SESSION['favorites'][] = $bookId;
            $_SESSION['flash_message'] = "'{$bookTitle}' 이(가) 보관함에 담아졌습니다.";
        }
    }

    // 장바구니 버튼 클릭 시 세션의 cart 배열에 도서 ID와 수량 저장 또는 수량 증가
    if (isset($_POST['add_to_cart'])) {
        $cart = $_SESSION['cart'] ?? [];

        if (isset($cart[$bookId])) {
            $cart[$bookId] += 1;  // 이미 있으면 수량 1 증가
        } else {
            $cart[$bookId] = 1;   // 새로 추가
        }

        $_SESSION['cart'] = $cart;
        $_SESSION['flash_message'] = "'{$bookTitle}' 이(가) 장바구니에 담겨졌습니다.";
    }

    // POST 처리 후 현재 페이지로 리다이렉트 (새로고침 시 중복 처리 방지)
    header("Location: " . $_SERVER['REQUEST_URI']);
    exit();
}
?>

<!DOCTYPE html>
<html lang="ko">

<head>
    <meta charset="UTF-8">
    <title><?php echo htmlspecialchars($categoryName) ?: '카테고리'; ?> - 도서 목록</title>
    <link rel="stylesheet" href="css/styles.css">
</head>
<style>
    body {
        margin: 0 auto;
        font-family: Arial, sans-serif;
        background-color: #f3f4f6;
        width: 1280px;
    }
</style>
<body>
    <!-- 헤더 영역 -->
    <div class="header page-section">
        <div class="logo">
            <a href="index.php">ReadUS</a>
        </div>
        <!-- 검색 바 -->
        <div class="search-bar">
            <form action="search.php" method="GET">
                <input type="text" name="q" placeholder="도서를 검색하세요!" required />
                <button type="submit" class="search-btn">검색</button>
            </form>
        </div>

        <!-- 사용자 메뉴 -->
        <div class="menu">
            <?php if (isset($_SESSION['loggedin']) && $_SESSION['loggedin'] === true): ?>
                <span class="welcome-message"><?php echo htmlspecialchars($_SESSION['username']); ?>님</span>
                <a href="edit-profile.php" title="회원정보 수정">⚙️</a>
                <a href="logout.php" title="로그아웃">🚪</a>
                <a href="mypage.php" title="MyPage">📚</a>
                <a href="wishlist.php" title="보관함">💖</a>
                <a href="cart.php" title="장바구니">🛒</a>
            <?php else: ?>
                <a href="login.php" title="로그인">🔐</a>
                <a href="signup.php" title="회원가입">👤</a>
            <?php endif; ?>
        </div>
    </div>

    <!-- 플래시 메시지 출력 (찜하기/장바구니 추가 후 메시지) -->
    <?php if (isset($_SESSION['flash_message'])): ?>
        <div class="flash-message">
            <?php
            echo htmlspecialchars($_SESSION['flash_message']);
            unset($_SESSION['flash_message']);  // 메시지 출력 후 제거
            ?>
        </div>
    <?php endif; ?>

    <!-- 카테고리 버튼 모음 -->
    <div class="category-buttons">
        <a href="category.php?name=경제+%2F+투자">경영 / 투자</a>
        <a href="category.php?name=IT+%2F+프로그래밍">IT / 프로그래밍</a>
        <a href="category.php?name=에세이">에세이</a>
        <a href="category.php?name=패션+%2F+뷰티">패션/뷰티</a>
        <a href="category.php?name=건강+%2F+운동">건강/운동</a>
        <a href="category.php?name=만화+%2F+웹툰">만화/웹툰</a>
    </div>

    <!-- 도서 목록 본문 -->
    <main>
        <?php if ($categoryName && $categoryId === null): ?>
            <!-- 카테고리명이 있으나 DB에 없을 때 -->
            <p>존재하지 않는 카테고리입니다.</p>
        <?php elseif (empty($books)): ?>
            <!-- 도서가 없을 때 -->
            <p>도서가 없습니다.</p>
        <?php else: ?>
            <!-- 도서가 있을 경우 도서 항목 반복 출력 -->
            <?php foreach ($books as $book): ?>
                <?php
                // 찜하기 및 장바구니 상태 확인
                $isFavorite = in_array($book['id'], $_SESSION['favorites'] ?? []);
                $isInCart = isset($_SESSION['cart'][$book['id']]);
                $imgFileName = $book['cover_image'] ?? 'default-book-cover.jpg';  // 표지 이미지 파일명
                $baseImgPath = "/~i2351013/ReadUS/img/";  // 이미지 기본 경로
                ?>
                <div class="book-item">
                    <div class="book-image">
                        <img src="<?php echo $baseImgPath . urlencode($imgFileName); ?>" alt="책 표지"
                            onerror="this.onerror=null; this.src='<?php echo $baseImgPath; ?>default-book-cover.jpg';">
                    </div>
                    <div class="book-info">
                        <div class="book-title"><?php echo htmlspecialchars($book['title']); ?></div>
                        <div class="book-author">저자: <?php echo htmlspecialchars($book['author']); ?></div>
                        <div class="book-publisher">출판사: <?php echo htmlspecialchars($book['publisher']); ?></div>
                        <div class="book-date">출간일: <?php echo htmlspecialchars($book['published_date']); ?></div>
                        <div class="book-description"><?php echo nl2br(htmlspecialchars($book['description'])); ?></div>
                        <div class="book-price">가격: <?php echo htmlspecialchars($book['price']); ?> 원</div>

                        <!-- 찜하기 및 장바구니 추가 폼 -->
                        <form action="" method="POST" class="icons">
                            <input type="hidden" name="book_id" value="<?php echo $book['id']; ?>">
                            <button type="submit" name="add_to_favorites" title="찜하기">❤️</button>
                            <button type="submit" name="add_to_cart" title="장바구니에 담기">🛒</button>
                        </form>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </main>

    <!-- 푸터 -->
    <div class="footer-box page-section">
        <div class="footer-item">안산대학교</div>
        <div class="footer-item">010-0000-0000</div>
        <div class="footer-item">shinyourim@ansan.ac.kr</div>
    </div>
</body>

</html>
