<?php
session_start();
include('db.php');

// 세션에서 장바구니 데이터를 가져옴
$cart = $_SESSION['cart'] ?? [];
$booksInCart = [];
$totalPrice = 0;

// 수량 변경 또는 삭제 요청 처리
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // 수량 변경 처리
    if (isset($_POST['update_quantity'])) {
        $bookId = (int)$_POST['book_id'];
        $quantity = max(1, (int)$_POST['quantity']); // 수량 최소 1로 제한

        if (isset($cart[$bookId])) {
            $cart[$bookId] = $quantity;    // 장바구니 수량 업데이트
            $_SESSION['cart'] = $cart;     // 세션에 저장
            $_SESSION['flash_message'] = "수량이 업데이트 되었습니다.";
        }
    }

    // 아이템 삭제 처리
    if (isset($_POST['remove_item'])) {
        $bookId = (int)$_POST['book_id'];
        if (isset($cart[$bookId])) {
            unset($cart[$bookId]);         // 장바구니에서 해당 도서 삭제
            $_SESSION['cart'] = $cart;     // 세션에 저장
            $_SESSION['flash_message'] = "장바구니에서 삭제되었습니다.";
        }
    }

    // 처리 후 새로고침 방지를 위해 리다이렉트
    header("Location: cart.php");
    exit();
}

// 장바구니에 담긴 도서 정보 조회
if (!empty($cart)) {
    // 장바구니에 담긴 도서 ID들로 SQL IN 절 생성
    $placeholders = implode(',', array_fill(0, count($cart), '?'));
    $types = str_repeat('i', count($cart));
    $stmt = $conn->prepare("SELECT * FROM books WHERE id IN ($placeholders)");

    $ids = array_keys($cart);
    $params = [];
    $params[] = &$types;    // 첫 번째 인자는 타입 문자열
    foreach ($ids as $key => $id) {
        $params[] = &$ids[$key];   // 각 도서 ID를 참조로 배열에 추가
    }

    // 동적으로 파라미터 바인딩
    call_user_func_array([$stmt, 'bind_param'], $params);

    $stmt->execute();
    $result = $stmt->get_result();

    // 결과에서 도서 정보를 가져와 수량, 소계, 총합 계산
    while ($row = $result->fetch_assoc()) {
        $row['quantity'] = $cart[$row['id']];
        $row['subtotal'] = $row['price'] * $row['quantity'];
        $totalPrice += $row['subtotal'];
        $booksInCart[] = $row;
    }
    $stmt->close();
}
?>

<!DOCTYPE html>
<html lang="ko">

<head>
    <meta charset="UTF-8">
    <title>장바구니 - ReadUS</title>
    <link rel="stylesheet" href="css/styles.css">
</head>
<style>
    body {
        margin: 0 auto;
        font-family: Arial, sans-serif;
        background-color: #f3f4f6;
        width: 1280px;
    }
</style>
<body>
    <!-- 헤더 영역 -->
    <div class="header page-section">
        <div class="logo">
            <a href="index.php">ReadUS</a>
        </div>
        <div class="menu">
            <?php if (isset($_SESSION['loggedin']) && $_SESSION['loggedin'] === true): ?>
                <!-- 로그인 상태일 때 환영 메시지와 메뉴 -->
                <span class="welcome-message"><?php echo htmlspecialchars($_SESSION['username']); ?>님</span>
                <a href="edit-profile.php" title="회원정보 수정">⚙️</a>
                <a href="logout.php" title="로그아웃">🚪</a>
                <a href="mypage.php" title="MyPage">📚</a>
                <a href="wishlist.php" title="보관함">💖</a>
                <a href="cart.php" title="장바구니">🛒</a>
            <?php else: ?>
                <!-- 비로그인 상태일 때 로그인/회원가입 링크 -->
                <a href="login.php" title="로그인">🔐</a>
                <a href="signup.php" title="회원가입">👤</a>
            <?php endif; ?>
        </div>
    </div>

    <!-- 플래시 메시지 출력 -->
    <?php if (isset($_SESSION['flash_message'])): ?>
        <div class="flash-message">
            <?php
            echo htmlspecialchars($_SESSION['flash_message']); // 메시지 출력
            unset($_SESSION['flash_message']);                // 메시지 삭제
            ?>
        </div>
    <?php endif; ?>

    <main>
        <h2>장바구니</h2>

        <!-- 장바구니가 비었을 경우 -->
        <?php if (empty($booksInCart)): ?>
            <p>장바구니에 담긴 도서가 없습니다.</p>
            <a href="index.php">도서 목록으로 돌아가기</a>
        <?php else: ?>
            <!-- 장바구니 도서 목록 테이블 -->
            <table class="cart-table">
                <thead>
                    <tr>
                        <th>책 표지</th>
                        <th>도서명</th>
                        <th>저자</th>
                        <th>가격</th>
                        <th>수량</th>
                        <th>합계</th>
                        <th>삭제</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($booksInCart as $book): ?>
                        <tr>
                            <td>
                                <!-- 도서 표지 이미지 -->
                                <img src="img/<?php echo htmlspecialchars($book['cover_image'] ?: 'default-book-cover.jpg'); ?>" alt="책 표지" style="width: 60px;">
                            </td>
                            <td><?php echo htmlspecialchars($book['title']); ?></td>
                            <td><?php echo htmlspecialchars($book['author']); ?></td>
                            <td><?php echo number_format($book['price']); ?> 원</td>
                            <td>
                                <!-- 수량 변경 폼 -->
                                <form method="POST" style="display: inline;">
                                    <input type="hidden" name="book_id" value="<?php echo $book['id']; ?>">
                                    <input type="number" name="quantity" value="<?php echo $book['quantity']; ?>" min="1" style="width: 50px;">
                                    <button type="submit" name="update_quantity">변경</button>
                                </form>
                            </td>
                            <td><?php echo number_format($book['subtotal']); ?> 원</td>
                            <td>
                                <!-- 삭제 폼 -->
                                <form method="POST" onsubmit="return confirm('정말 삭제하시겠습니까?');">
                                    <input type="hidden" name="book_id" value="<?php echo $book['id']; ?>">
                                    <button type="submit" name="remove_item">삭제</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
                <tfoot>
                    <tr>
                        <td colspan="5" style="text-align:right; font-weight:bold;">총 합계:</td>
                        <td colspan="2" style="font-weight:bold;"><?php echo number_format($totalPrice); ?> 원</td>
                    </tr>
                </tfoot>
            </table>

            <!-- 구매 계속하기 및 결제하기 버튼 -->
<div style="margin-top: 30px; text-align: center;">
    <a href="category.php" style="
        display: inline-block;
        background-color: #6a4fff;
        color: white;
        padding: 10px 25px;
        margin: 10px;
        border-radius: 8px;
        font-weight: 600;
        font-size: 1rem;
        text-decoration: none;
        transition: background-color 0.3s ease;
    "
    onmouseover="this.style.backgroundColor='#533dcc';" 
    onmouseout="this.style.backgroundColor='#6a4fff';"
    >도서 계속 구매하기</a>

    <a href="checkout.php" style="
        display: inline-block;
        background-color: #6a4fff;
        color: white;
        padding: 10px 25px;
        margin: 10px;
        border-radius: 8px;
        font-weight: 600;
        font-size: 1rem;
        text-decoration: none;
        transition: background-color 0.3s ease;
    "
    onmouseover="this.style.backgroundColor='#533dcc';" 
    onmouseout="this.style.backgroundColor='#6a4fff';"
    >결제하기</a>
</div>

        <?php endif; ?>
    </main>

    <!-- 푸터 영역 -->
    <div class="footer-box page-section">
        <div class="footer-item">안산대학교</div>
        <div class="footer-item">010-0000-0000</div>
        <div class="footer-item">shinyourim@ansan.ac.kr</div>
    </div>
</body>

</html>
