package com.example.smoking;

import android.Manifest;
import android.app.Dialog;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.location.Address;
import android.location.Geocoder;
import android.location.Location;
import android.os.Bundle;
import android.provider.MediaStore;
import android.util.Base64;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.Toast;

import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.DialogFragment;

import com.android.volley.AuthFailureError;
import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.toolbox.StringRequest;
import com.android.volley.toolbox.Volley;
import com.google.android.gms.location.FusedLocationProviderClient;
import com.google.android.gms.location.LocationServices;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

public class ReportFragment extends DialogFragment {

    private ActivityResultLauncher<Intent> takePictureLauncher;
    private ImageView imageView;
    private Bitmap capturedPhoto;
    private FusedLocationProviderClient fusedLocationClient;

    private double currentLatitude = 0.0;
    private double currentLongitude = 0.0;
    private String currentAddress = "";

    private final double fallbackLatitude = 37.2985;
    private final double fallbackLongitude = 126.8342;
    private final String fallbackAddress = "경기도 안산시 상록구 안산대학로 155 (일동)";

    public ReportFragment() {}

    private void uploadReportWithPhoto(String address, double latitude, double longitude, Bitmap photo) {
        String url = "https://infognu.ansan.ac.kr/~i2151046/smoking/upload_report.php";

        RequestQueue queue = Volley.newRequestQueue(requireContext());

        StringRequest stringRequest = new StringRequest(Request.Method.POST, url,
                response -> {
                    Log.d("UPLOAD", "서버 응답: " + response);
                    Toast.makeText(getContext(), "제보가 완료되었습니다.", Toast.LENGTH_SHORT).show();
                },
                error -> {
                    Log.e("UPLOAD", "에러: " + error.getMessage());
                    Toast.makeText(getContext(), "제보에 실패했습니다.", Toast.LENGTH_SHORT).show();
                }) {

            @Override
            protected Map<String, String> getParams() throws AuthFailureError {
                Map<String, String> params = new HashMap<>();
                params.put("address", address);
                params.put("latitude", String.valueOf(latitude));
                params.put("longitude", String.valueOf(longitude));
                params.put("image", encodeBitmapToBase64(photo));
                return params;
            }
        };

        queue.add(stringRequest);
    }

    private String encodeBitmapToBase64(Bitmap bitmap) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        bitmap.compress(Bitmap.CompressFormat.JPEG, 60, baos); // 압축률 60%
        byte[] imageBytes = baos.toByteArray();
        return Base64.encodeToString(imageBytes, Base64.NO_WRAP);
    }

    private void requestPermissions() {
        if (ContextCompat.checkSelfPermission(requireContext(), Manifest.permission.CAMERA)
                != PackageManager.PERMISSION_GRANTED ||
                ContextCompat.checkSelfPermission(requireContext(), Manifest.permission.ACCESS_FINE_LOCATION)
                        != PackageManager.PERMISSION_GRANTED) {
            ActivityCompat.requestPermissions(requireActivity(),
                    new String[]{Manifest.permission.CAMERA, Manifest.permission.ACCESS_FINE_LOCATION}, 100);
        }
    }

    @Override
    public void onStart() {
        super.onStart();
        if (getDialog() != null && getDialog().getWindow() != null) {
            Window window = getDialog().getWindow();
            window.setLayout(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
            window.setBackgroundDrawable(new ColorDrawable(Color.parseColor("#AAFFFFFF")));
        }
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater,
                             @Nullable ViewGroup container,
                             @Nullable Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fragment_report, container, false);

        requestPermissions();

        imageView = view.findViewById(R.id.image_report_photo);
        fusedLocationClient = LocationServices.getFusedLocationProviderClient(requireActivity());

        Button closeBtn = view.findViewById(R.id.btn_close_report);
        closeBtn.setOnClickListener(v -> dismiss());

        Button btnTakePhoto = view.findViewById(R.id.btn_take_photo);
        takePictureLauncher = registerForActivityResult(
                new ActivityResultContracts.StartActivityForResult(),
                result -> {
                    if (result.getResultCode() == requireActivity().RESULT_OK) {
                        Intent data = result.getData();
                        if (data != null && data.getExtras() != null) {
                            capturedPhoto = (Bitmap) data.getExtras().get("data");
                            imageView.setImageBitmap(capturedPhoto);
                        }
                    }
                });

        btnTakePhoto.setOnClickListener(v -> {
            Intent takePictureIntent = new Intent(MediaStore.ACTION_IMAGE_CAPTURE);
            if (takePictureIntent.resolveActivity(requireActivity().getPackageManager()) != null) {
                takePictureLauncher.launch(takePictureIntent);
            }
        });

        Button btnLocation = view.findViewById(R.id.btn_get_location);
        Button btnAddress = view.findViewById(R.id.btn_address_display);

        btnLocation.setOnClickListener(v -> {
            if (ContextCompat.checkSelfPermission(requireContext(), Manifest.permission.ACCESS_FINE_LOCATION)
                    == PackageManager.PERMISSION_GRANTED) {

                fusedLocationClient.getLastLocation()
                        .addOnSuccessListener(location -> {
                            double latitude = fallbackLatitude;
                            double longitude = fallbackLongitude;
                            String addressText = fallbackAddress;

                            if (location != null) {
                                latitude = location.getLatitude();
                                longitude = location.getLongitude();

                                Geocoder geocoder = new Geocoder(requireContext(), Locale.KOREA);
                                try {
                                    List<Address> addresses = geocoder.getFromLocation(latitude, longitude, 1);
                                    if (addresses != null && !addresses.isEmpty()) {
                                        addressText = addresses.get(0).getAddressLine(0);
                                    }
                                } catch (IOException e) {
                                    addressText = "지오코딩 오류";
                                }
                            }

                            currentLatitude = latitude;
                            currentLongitude = longitude;
                            currentAddress = addressText;

                            btnAddress.setText(addressText + "\n위도: " + latitude + "\n경도: " + longitude);
                        });

            } else {
                Toast.makeText(getContext(), "위치 권한이 필요합니다.", Toast.LENGTH_SHORT).show();
                requestPermissions();
            }
        });

        Button btnSubmit = view.findViewById(R.id.btn_submit_report);
        btnSubmit.setOnClickListener(v -> {
            if (currentAddress.isEmpty() || currentLatitude == 0.0 || currentLongitude == 0.0) {
                Toast.makeText(getContext(), "위치 정보를 먼저 가져와 주세요.", Toast.LENGTH_SHORT).show();
                return;
            }
            if (capturedPhoto == null) {
                Toast.makeText(getContext(), "사진을 먼저 촬영해 주세요.", Toast.LENGTH_SHORT).show();
                return;
            }

            uploadReportWithPhoto(currentAddress, currentLatitude, currentLongitude, capturedPhoto);
        });

        return view;
    }

    @Override
    public Dialog onCreateDialog(Bundle savedInstanceState) {
        Dialog dialog = super.onCreateDialog(savedInstanceState);
        dialog.getWindow().requestFeature(Window.FEATURE_NO_TITLE);
        return dialog;
    }
}
