package com.example.smoking;

import android.content.SharedPreferences;
import android.graphics.Color;
import android.os.Bundle;
import android.widget.Button;
import android.widget.TextView;
import android.widget.Toast;

import androidx.appcompat.app.AppCompatActivity;

import com.android.volley.Request;
import com.android.volley.toolbox.StringRequest;
import com.android.volley.toolbox.Volley;
import com.prolificinteractive.materialcalendarview.CalendarDay;
import com.prolificinteractive.materialcalendarview.MaterialCalendarView;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.*;

public class HabbitActivity extends AppCompatActivity {

    private MaterialCalendarView calendarView;
    private TextView textCount;
    private int count = 0;

    private CalendarDay selectedDay;
    private final Map<CalendarDay, Integer> habitMap = new HashMap<>();

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_habbit);

        calendarView = findViewById(R.id.calendarView);
        textCount = findViewById(R.id.text_count);
        Button btnPlus = findViewById(R.id.btn_plus);
        Button btnMinus = findViewById(R.id.btn_minus);
        Button btnSubmit = findViewById(R.id.btn_submit);

        // 오늘 날짜 기본 선택
        selectedDay = CalendarDay.today();
        calendarView.setDateSelected(selectedDay.getDate(), true);
        TextView textSelectedCount = findViewById(R.id.text_selected_count);

        calendarView.setOnDateChangedListener((widget, date, selected) -> {
            selectedDay = date;
            loadHabitFromServer(getDateStr(selectedDay));

            int pins = habitMap.getOrDefault(selectedDay, 0);
            textSelectedCount.setText("핀 개수: " + pins + "개");
        });

        btnPlus.setOnClickListener(v -> {
            count++;
            updateCount();
        });

        btnMinus.setOnClickListener(v -> {
            if (count > 0) count--;
            updateCount();
        });

        btnSubmit.setOnClickListener(v -> saveHabitToServer());

        // 전체 기록 불러오기
        loadAllHabitsFromServer();
        loadHabitFromServer(getDateStr(selectedDay));
    }

    private void updateCount() {
        textCount.setText(String.valueOf(count));
    }

    private String getDateStr(CalendarDay day) {
        Calendar cal = Calendar.getInstance();
        cal.set(day.getYear(), day.getMonth(), day.getDay());
        return new java.text.SimpleDateFormat("yyyy-MM-dd", Locale.getDefault()).format(cal.getTime());
    }

    private void saveHabitToServer() {
        SharedPreferences prefs = getSharedPreferences("MyAppPreferences", MODE_PRIVATE);
        String userId = prefs.getString("userId", null);
        if (userId == null || selectedDay == null) return;

        String url = "https://infognu.ansan.ac.kr/~i2151046/smoking/insert_or_update_habit.php";
        StringRequest request = new StringRequest(Request.Method.POST, url,
                response -> {
                    try {
                        JSONObject json = new JSONObject(response);
                        if ("success".equals(json.getString("result"))) {
                            habitMap.put(selectedDay, count);
                            refreshCalendar();
                            Toast.makeText(this, "저장 완료", Toast.LENGTH_SHORT).show();
                        }
                    } catch (JSONException e) {
                        Toast.makeText(this, "저장 실패", Toast.LENGTH_SHORT).show();
                    }
                },
                error -> Toast.makeText(this, "서버 오류", Toast.LENGTH_SHORT).show()
        ) {
            @Override
            protected Map<String, String> getParams() {
                Map<String, String> params = new HashMap<>();
                params.put("사용자아이디", userId);
                params.put("기록날짜", getDateStr(selectedDay));
                params.put("핀갯수", String.valueOf(count));
                return params;
            }
        };

        Volley.newRequestQueue(this).add(request);
    }

    private void loadHabitFromServer(String dateStr) {
        SharedPreferences prefs = getSharedPreferences("MyAppPreferences", MODE_PRIVATE);
        String userId = prefs.getString("userId", null);
        if (userId == null || selectedDay == null) return;

        String url = "https://infognu.ansan.ac.kr/~i2151046/smoking/get_habit_by_date.php";
        StringRequest request = new StringRequest(Request.Method.POST, url,
                response -> {
                    try {
                        JSONObject json = new JSONObject(response);
                        if ("success".equals(json.getString("result"))) {
                            count = json.getInt("핀갯수");
                            habitMap.put(selectedDay, count);
                        } else {
                            count = 0;
                        }
                        updateCount();
                        refreshCalendar();
                    } catch (JSONException e) {
                        Toast.makeText(this, "불러오기 실패", Toast.LENGTH_SHORT).show();
                    }
                },
                error -> Toast.makeText(this, "서버 오류", Toast.LENGTH_SHORT).show()
        ) {
            @Override
            protected Map<String, String> getParams() {
                Map<String, String> params = new HashMap<>();
                params.put("사용자아이디", userId);
                params.put("기록날짜", dateStr);
                return params;
            }
        };

        Volley.newRequestQueue(this).add(request);
    }

    private void loadAllHabitsFromServer() {
        SharedPreferences prefs = getSharedPreferences("MyAppPreferences", MODE_PRIVATE);
        String userId = prefs.getString("userId", null);
        if (userId == null) return;

        String url = "https://infognu.ansan.ac.kr/~i2151046/smoking/get_all_habits.php";
        StringRequest request = new StringRequest(Request.Method.POST, url,
                response -> {
                    try {
                        JSONObject json = new JSONObject(response);
                        if ("success".equals(json.getString("result"))) {
                            JSONArray data = json.getJSONArray("data");
                            for (int i = 0; i < data.length(); i++) {
                                JSONObject obj = data.getJSONObject(i);
                                String dateStr = obj.getString("기록날짜");
                                int pin = obj.getInt("핀갯수");

                                String[] parts = dateStr.split("-");
                                int y = Integer.parseInt(parts[0]);
                                int m = Integer.parseInt(parts[1]) - 1;
                                int d = Integer.parseInt(parts[2]);

                                CalendarDay day = CalendarDay.from(y, m, d);
                                habitMap.put(day, pin);
                            }
                            refreshCalendar();
                        }
                    } catch (JSONException e) {
                        Toast.makeText(this, "전체 기록 불러오기 실패", Toast.LENGTH_SHORT).show();
                    }
                },
                error -> Toast.makeText(this, "전체 기록 네트워크 오류", Toast.LENGTH_SHORT).show()
        ) {
            @Override
            protected Map<String, String> getParams() {
                Map<String, String> params = new HashMap<>();
                params.put("사용자아이디", userId);
                return params;
            }
        };

        Volley.newRequestQueue(this).add(request);
    }

    private void refreshCalendar() {
        calendarView.removeDecorators();

        Set<CalendarDay> recordedDates = habitMap.keySet();
        calendarView.addDecorator(new DotDecorator(recordedDates, Color.RED));
    }
}
