<?php
require_once "../../includes/db_connect.php"; // DB 연결
require_once "../../includes/islogin.php";    // 로그인 확인

$message = "";
$restaurant_name_display = "식당 정보 없음";
$ordered_food_names_display_string = ""; // 화면 표시용 주문 음식 전체 목록 (쉼표 구분 문자열)

// URL 파라미터
$current_order_id = 0;
$current_restaurant_id = 0;


// 1. GET 파라미터 받기
if (isset($_GET['order_id']) && isset($_GET['restaurant_id'])) {
    $current_order_id = (int)$_GET['order_id'];
    $current_restaurant_id = (int)$_GET['restaurant_id'];

    if ($current_order_id <= 0 || $current_restaurant_id <= 0) {
        die("잘못된 파라미터입니다. (order_id 또는 restaurant_id 누락/오류)");
    }

    $stmt_check_order_owner = $conn->prepare("SELECT user_id FROM Orders WHERE order_id = ? AND user_id = ?");
    if ($stmt_check_order_owner) {
        $stmt_check_order_owner->bind_param("ii", $current_order_id, $_SESSION['user_id']);
        $stmt_check_order_owner->execute();
        if ($stmt_check_order_owner->get_result()->num_rows === 0) {
            die("주문 내역에 접근할 권한이 없습니다.");
        }
        $stmt_check_order_owner->close();
    } else {
        die("DB 오류 (주문 소유자 확인 실패): " . $conn->error);
    }


    // 이미 이 '주문 전체'에 대해 리뷰를 작성했는지 확인
    $stmt_check_review = $conn->prepare("SELECT review_id FROM Reviews WHERE order_id = ? AND user_id = ? AND food_id IS NULL AND order_item_id IS NULL");
    if($stmt_check_review){
        $stmt_check_review->bind_param("ii", $current_order_id, $_SESSION['user_id']);
        $stmt_check_review->execute();
        $result_check_review = $stmt_check_review->get_result();
        if($result_check_review->num_rows > 0){
            $message = "이미 이 주문에 대한 리뷰를 작성하셨습니다.";
        }
        $stmt_check_review->close();
    }


    // 식당 이름 가져오기
    $stmt_restaurant = $conn->prepare("SELECT name FROM Restaurants WHERE restaurant_id = ?");
    if ($stmt_restaurant) {
        $stmt_restaurant->bind_param("i", $current_restaurant_id);
        $stmt_restaurant->execute();
        $result_restaurant = $stmt_restaurant->get_result();
        if ($result_restaurant->num_rows > 0) {
            $restaurant_data = $result_restaurant->fetch_assoc();
            $restaurant_name_display = $restaurant_data['name'];
        } else {

        }
        $stmt_restaurant->close();
    } else {
        die("식당 정보 조회 오류: " . $conn->error);
    }

    // 해당 주문의 '모든' 음식 아이템 이름 가져오기
    $food_names_array = [];
    $stmt_all_items_in_order = $conn->prepare(
        "SELECT f.name AS food_item_name
         FROM OrderItems oi
         JOIN Foods f ON oi.food_id = f.food_id
         WHERE oi.order_id = ?
         ORDER BY oi.order_item_id ASC"
    );
    if ($stmt_all_items_in_order) {
        $stmt_all_items_in_order->bind_param("i", $current_order_id);
        $stmt_all_items_in_order->execute();
        $result_all_items_in_order = $stmt_all_items_in_order->get_result();
        while ($item_row_in_order = $result_all_items_in_order->fetch_assoc()) {
            $food_names_array[] = htmlspecialchars($item_row_in_order['food_item_name']);
        }
        $stmt_all_items_in_order->close();
        if (!empty($food_names_array)) {
            $ordered_food_names_display_string = implode(', ', $food_names_array);
        } else {
            $ordered_food_names_display_string = "주문된 음식 정보를 가져올 수 없습니다.";
        }
    } else {
        $ordered_food_names_display_string = "음식 정보 조회 오류.";
        error_log("DB 오류 (주문 아이템 조회 준비 실패): " . $conn->error);
    }

} else {
    die("주문 정보가 정확히 전달되지 않았습니다. (order_id 또는 restaurant_id 누락)");
}


// 폼 제출 처리 (주문 전체에 대한 리뷰 저장)
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (strpos($message, "이미 이 주문에 대한 리뷰를 작성하셨습니다") !== false) {
        // 이미 리뷰를 작성했다는 메시지가 있으면 더 이상 진행하지 않음
    } elseif (!isset($_SESSION['user_id'])) {
        $message = "리뷰를 작성하려면 로그인이 필요합니다.";
    } else {
        $user_id = $_SESSION['user_id'];
        $p_order_id = isset($_POST['order_id']) ? (int)$_POST['order_id'] : 0;
        $p_restaurant_id = isset($_POST['restaurant_id']) ? (int)$_POST['restaurant_id'] : 0;

        if ($p_order_id !== $current_order_id || $p_restaurant_id !== $current_restaurant_id) {
            die("요청 정보가 일치하지 않습니다. (ID 불일치)");
        }

        $rating = isset($_POST['rating']) ? (int)$_POST['rating'] : 0;
        $comment = isset($_POST['comment']) ? trim($_POST['comment']) : "";
        $photo_url = null;

        if ($rating < 1 || $rating > 5) {
            $message = "별점을 선택해주세요 (1-5점).";
        } elseif (empty($comment)) {
            $message = "리뷰 내용을 입력해주세요.";
        } else {
            // 사진 업로드 처리
            if (isset($_FILES['photo']) && $_FILES['photo']['error'] == UPLOAD_ERR_OK) {
                $document_root = $_SERVER['DOCUMENT_ROOT'];
                $upload_dir_filesystem = $document_root . '/Capstone/assets/images/'; // 실제 파일 시스템 경로
                
                // DB에 저장하고 HTML에서 사용할 웹 경로 (앞에 / 를 붙여 웹 루트 기준)
                $upload_dir_web = '/Capstone/assets/images/';
            
                if (!is_dir($upload_dir_filesystem)) {
                    if (!@mkdir($upload_dir_filesystem, 0775, true) && !is_dir($upload_dir_filesystem)) {
                         error_log('Failed to create directory: ' . $upload_dir_filesystem);
                         $message = "업로드 디렉토리 생성 실패. 리뷰는 사진 없이 저장됩니다.";
                         $photo_url = null;
                    }
                }
            
                if (is_dir($upload_dir_filesystem)) { // 디렉토리가 성공적으로 존재하거나 생성된 경우에만 진행
                    $file_extension = strtolower(pathinfo($_FILES['photo']['name'], PATHINFO_EXTENSION));
                    $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif'];
            
                    if (in_array($file_extension, $allowed_extensions)) {
                        $photo_name = 'order_rev_' . $p_order_id . '_' . uniqid() . '.' . $file_extension;
                        $target_file_filesystem = $upload_dir_filesystem . '/' . $photo_name; // 실제 저장될 파일 시스템 경로
            
                        if (move_uploaded_file($_FILES['photo']['tmp_name'], $target_file_filesystem)) {
                            // DB에는 웹에서 접근 가능한 웹 경로 저장
                            $photo_url = $upload_dir_web . $photo_name; 
                        } else {
                            error_log('Failed to move uploaded file to: ' . $target_file_filesystem);
                            $message = "사진 업로드 실패 (파일 이동 오류). 리뷰는 사진 없이 저장됩니다.";
                            $photo_url = null;
                        }
                    } else {
                        $message = "지원하지 않는 파일 형식입니다 (jpg, jpeg, png, gif만 가능). 리뷰는 사진 없이 저장됩니다.";
                        $photo_url = null;
                    }
                }
            } else {
                // 파일이 업로드되지 않았거나 오류 발생
                if (isset($_FILES['photo']) && $_FILES['photo']['error'] != UPLOAD_ERR_NO_FILE) {
                    // UPLOAD_ERR_NO_FILE (파일 첨부 안 함)이 아닌 다른 오류일 경우 메시지 설정
                    $message = "사진 업로드 중 오류 발생 (오류 코드: " . $_FILES['photo']['error'] . "). 리뷰는 사진 없이 저장됩니다.";
                }
            }

            // Reviews 테이블에 주문 전체에 대한 리뷰 저장
            $sql_insert = "INSERT INTO Reviews (user_id, restaurant_id, order_id, food_id, order_item_id, rating, comment, photo_url, created_at)
                           VALUES (?, ?, ?, NULL, NULL, ?, ?, ?, NOW())";

            if ($stmt_insert = $conn->prepare($sql_insert)) {
                $stmt_insert->bind_param("iiisss", $user_id, $p_restaurant_id, $p_order_id, $rating, $comment, $photo_url);
                if ($stmt_insert->execute()) {
                    $message = "리뷰가 성공적으로 등록되었습니다!";
                } else {
                    if ($stmt_insert->errno == 1062) {
                        $message = "이미 이 주문에 대한 리뷰를 작성하셨습니다.";
                    } else {
                        $message = "리뷰 등록에 실패했습니다: " . $stmt_insert->error;
                    }
                }
                $stmt_insert->close();
            } else {
                $message = "데이터베이스 오류 (INSERT 준비 실패): " . $conn->error;
            }
        }
    }
}
$conn->close();
?>

<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <title>리뷰 작성 - <?php echo htmlspecialchars($restaurant_name_display); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css" />
    <style>
        html, body { margin: 0; padding: 0; font-family: sans-serif; background-color: #f4f4f4; }
        .app-container { max-width: 500px; margin: 0 auto; display: flex; flex-direction: column; min-height: 100vh; background-color: white; box-shadow: 0 0 10px rgba(0,0,0,0.1); }
        .main-content { flex: 1; overflow-y: auto; padding: 15px; }
        .header { display: flex; align-items: center; justify-content: space-between; padding: 10px 15px; border-bottom: 1px solid #ddd; background-color: #fff; }
        .back-button { text-decoration: none; color: #333; font-size: 1.5em; }
        .review-title { font-weight: bold; font-size: 1.2em; flex: 1; text-align: center; }

        .review-target-info {
            padding: 15px;
            background-color: #f0f5ff;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
        }
        .review-target-info .restaurant {
            font-weight: bold;
            font-size: 1.2em;
            color: #333;
            margin-bottom: 5px;
        }
        .review-target-info .food-display {
            font-size: 0.95em;
            color: #007bff;
            line-height: 1.5;
            word-break: keep-all;
        }
        .review-target-info .food-display span:not(:last-child):after {
            content: ", ";
            color: #007bff;
        }

        .form-group { margin-bottom: 15px; }
        .form-group label { display: block; margin-bottom: 5px; font-weight: bold; font-size: 0.9em; }
        .form-group textarea { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; font-size: 1em; height: 120px; resize: vertical; }
        .stars { display: flex; justify-content: center; margin-bottom: 15px; }
        .stars i { font-size: 2.2em; margin: 0 5px; color: #ccc; cursor: pointer; transition: color 0.2s; }
        .stars i.selected, .stars i:hover { color: #ffc107; }
        .photo-button-label { display: inline-block; background: #e0e0e0; color: #333; padding: 10px 15px; border-radius: 4px; cursor: pointer; font-size: 0.9em; transition: background-color 0.2s; }
        .photo-button-label:hover { background: #d0d0d0; }
        .photo-button-label i { margin-right: 8px; }
        input[type="file"] { display: none; }
        .photo-preview { text-align: center; margin-top: 10px; }
        .photo-preview img { max-width: 100%; max-height: 250px; border-radius: 4px; border: 1px solid #ddd; }
        .photo-preview .no-photo { font-size:0.9em; color: #777; }
        .notice { font-size: 0.8em; color: #666; margin-top: 20px; padding: 10px; background-color: #f9f9f9; border-radius: 4px; border-left: 3px solid #007bff; }
        .submit-button { width: 100%; background: #007bff; color: white; font-size: 1.1em; padding: 12px 0; border: none; border-radius: 4px; cursor: pointer; transition: background-color 0.2s; margin-top:20px; }
        .submit-button:hover { background: #0056b3; }
        .submit-button:disabled { background-color: #ccc; cursor: not-allowed; }
        .message-area { padding: 10px; margin: 10px 0; border-radius: 4px; text-align: center; font-size: 0.9em;}
        .message-area.success { background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .message-area.error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .bottom-nav { display: flex; justify-content: space-around; align-items: center; padding: 10px 0; border-top: 1px solid #eee; background-color: white; flex-shrink: 0; position:sticky; bottom:0; }
        .bottom-nav a { color: #555; font-size: 1.5em; text-decoration: none; padding: 10px 15px; }
        .bottom-nav a.active, .bottom-nav a:hover { color: #007bff; }
    </style>
</head>
<body>
    <div class="app-container">
        <div class="header">
            <a href="javascript:history.back();" class="back-button"><i class="fas fa-arrow-left"></i></a>
            <div class="review-title">리뷰 작성</div>
            <div style="width: 24px;"></div>
        </div>

        <div class="main-content">
            <div class="review-target-info">
                <div class="restaurant"><?php echo htmlspecialchars($restaurant_name_display); ?></div>
                <div class="food-display">
                    <?php
                    if (!empty($food_names_array)) {
                        $display_foods = [];
                        foreach($food_names_array as $f_name) {
                            $display_foods[] = "<span>" . $f_name . "</span>";
                        }
                        echo implode("", $display_foods);
                    } else {
                        echo "<span>주문하신 음식 정보를 불러올 수 없습니다.</span>";
                    }
                    ?>
                </div>
            </div>

            <?php if (!empty($message)): ?>
                <div class="message-area <?php echo (strpos($message, "성공") !== false) ? 'success' : 'error'; ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <?php
            // 이미 리뷰를 작성했거나, 성공 메시지가 있다면 폼을 표시하지 않음.
            $show_form = true;
            if (strpos($message, "이미 이 주문에 대한 리뷰를 작성하셨습니다") !== false || strpos($message, "성공적으로 등록되었습니다!") !== false) {
                $show_form = false;
            }

            if ($show_form) :
            ?>
            <form method="POST" action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]) . '?order_id=' . $current_order_id . '&restaurant_id=' . $current_restaurant_id; ?>" enctype="multipart/form-data" id="reviewForm">
                <input type="hidden" name="order_id" value="<?php echo $current_order_id; ?>">
                <input type="hidden" name="restaurant_id" value="<?php echo $current_restaurant_id; ?>">
                <input type="hidden" name="rating" id="hidden-rating" value="0">

                <div class="form-group">
                    <label>별점을 선택해주세요:</label>
                    <div class="stars" id="star-rating">
                        <i class="far fa-star" data-value="1"></i><i class="far fa-star" data-value="2"></i><i class="far fa-star" data-value="3"></i><i class="far fa-star" data-value="4"></i><i class="far fa-star" data-value="5"></i>
                    </div>
                </div>
                <div class="form-group">
                    <label for="comment">리뷰 내용 (이 주문 전체에 대한 평가):</label>
                    <textarea id="comment" name="comment" placeholder="주문하신 메뉴들과 서비스에 대한 전체적인 리뷰를 남겨주세요." required></textarea>
                </div>
                <div class="form-group">
                    <label for="photo-input" class="photo-button-label"><i class="fas fa-camera"></i>사진 첨부 (선택 사항)</label>
                    <input type="file" id="photo-input" name="photo" accept="image/*">
                    <div class="photo-preview" id="photo-preview"><span class="no-photo">선택된 사진 없음</span></div>
                </div>
                <div class="notice">
                    솔직한 리뷰는 다른 사용자에게 큰 도움이 됩니다. 비방이나 욕설, 허위 사실 유포는 삼가주세요.
                </div>
                <button type="submit" class="submit-button" id="submitBtn">리뷰 등록</button>
            </form>
            <?php endif; ?>
        </div>

        <nav class="bottom-nav">
            <a href="../../main.php" class="active"><i class="fas fa-home"></i></a>
            <a href="../user/dibs.php"><i class="fas fa-heart"></i></a>
            <a href="../order/orderlist.php"><i class="fas fa-receipt"></i></a>
            <a href="../../auth/redirect-mypage.php"><i class="fas fa-user"></i></a>
        </nav>
    </div>

    <script>
        const starsContainer = document.getElementById("star-rating");
        const hiddenRatingInput = document.getElementById("hidden-rating");
        const reviewForm = document.getElementById('reviewForm');
        const submitBtn = document.getElementById('submitBtn');
        let currentRating = 0;

        if (starsContainer && hiddenRatingInput) {
            const stars = starsContainer.querySelectorAll("i");
            stars.forEach(star => {
                star.addEventListener("click", () => {
                    currentRating = parseInt(star.getAttribute("data-value"));
                    hiddenRatingInput.value = currentRating;
                    updateStarsVisual(stars, currentRating);
                });
                star.addEventListener("mouseover", () => {
                    const hoverValue = parseInt(star.getAttribute("data-value"));
                    updateStarsVisual(stars, hoverValue, true);
                });
            });
            starsContainer.addEventListener("mouseleave", () => {
                updateStarsVisual(stars, currentRating);
            });
        }

        function updateStarsVisual(starElements, ratingValue, isHover = false) {
            if(starElements){
                starElements.forEach(s => {
                    const sValue = parseInt(s.getAttribute("data-value"));
                    if (sValue <= ratingValue) {
                        s.classList.add("selected", "fas"); s.classList.remove("far");
                    } else {
                        s.classList.remove("selected", "fas"); s.classList.add("far");
                    }
                });
            }
        }

        const fileInput = document.getElementById('photo-input');
        const photoPreview = document.getElementById('photo-preview');

        if(fileInput && photoPreview){
            const noPhotoSpan = photoPreview.querySelector('.no-photo');
            fileInput.addEventListener('change', () => {
                const file = fileInput.files[0];
                if (file) {
                    const reader = new FileReader();
                    reader.onload = e => {
                        if(noPhotoSpan) noPhotoSpan.style.display = 'none';
                        photoPreview.innerHTML = `<img src="${e.target.result}" alt="미리보기">`;
                    };
                    reader.readAsDataURL(file);
                } else {
                    if(noPhotoSpan) noPhotoSpan.style.display = 'inline'; // 사진 선택 취소 시 다시 보이게
                    photoPreview.innerHTML = '<span class="no-photo">선택된 사진 없음</span>';
                }
            });
        }

        if(reviewForm && submitBtn){
            reviewForm.addEventListener('submit', function(event) {
                if (hiddenRatingInput && (hiddenRatingInput.value === "0" || currentRating === 0)) {
                    alert("별점을 선택해주세요.");
                    event.preventDefault(); return false;
                }
                const commentTextarea = document.getElementById('comment');
                if (commentTextarea && commentTextarea.value.trim() === "") {
                    alert("리뷰 내용을 입력해주세요.");
                    event.preventDefault(); commentTextarea.focus(); return false;
                }
                submitBtn.disabled = true;
                submitBtn.textContent = '등록 중...';
            });
        }
    </script>
    <?php if(isset($conn) && $conn instanceof mysqli) { $conn->close(); } ?>
</body>
</html>