<?php
require_once "../includes/islogin.php";
require_once "../includes/db_connect.php";

header('Content-Type: application/json'); // 응답을 JSON 형태로 설정

$response = ['success' => false, 'message' => '', 'liked' => false, 'new_like_count' => 0];

// 1. 사용자 로그인 상태 확인
if (!isset($_SESSION['user_id'])) {
    $response['message'] = '로그인이 필요합니다.';
    echo json_encode($response);
    exit;
}
$user_id = (int)$_SESSION['user_id'];

// 2. POST 요청 및 review_id 유효성 검사
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $response['message'] = '잘못된 요청 방식입니다.';
    echo json_encode($response);
    exit;
}

if (!isset($_POST['review_id']) || !filter_var($_POST['review_id'], FILTER_VALIDATE_INT) || ((int)$_POST['review_id'] <= 0) ) {
    $response['message'] = '잘못된 리뷰 ID입니다.';
    echo json_encode($response);
    exit;
}
$review_id = (int)$_POST['review_id'];

// 3. 데이터베이스 연결 확인 (db_connect.php에서 $conn 객체를 생성한다고 가정)
if (!$conn || $conn->connect_error) {
    $response['message'] = '데이터베이스 연결에 실패했습니다.';
    error_log("toggle_review_like.php: DB connection error - " . ($conn ? $conn->connect_error : 'conn object null'));
    echo json_encode($response);
    exit;
}

// 4. 리뷰 존재 여부 확인 (선택 사항이지만 권장)
$sql_check_review_exists = "SELECT review_id FROM Reviews WHERE review_id = ?";
if ($stmt_check_review = $conn->prepare($sql_check_review_exists)) {
    $stmt_check_review->bind_param("i", $review_id);
    $stmt_check_review->execute();
    $result_check_review = $stmt_check_review->get_result();
    if ($result_check_review->num_rows === 0) {
        $response['message'] = '존재하지 않는 리뷰입니다.';
        $stmt_check_review->close();
        echo json_encode($response);
        exit;
    }
    $stmt_check_review->close();
} else {
    $response['message'] = 'DB 오류: 리뷰 확인 중 오류가 발생했습니다.';
    error_log("toggle_review_like.php: Failed to prepare statement to check review existence - " . $conn->error);
    echo json_encode($response);
    exit;
}


// 5. 현재 좋아요 상태 확인 및 토글 처리
$conn->begin_transaction(); // 트랜잭션 시작

try {
    // 현재 좋아요 상태 확인
    $sql_check_like = "SELECT user_id FROM ReviewLikes WHERE user_id = ? AND review_id = ?";
    $stmt_check = $conn->prepare($sql_check_like);
    if (!$stmt_check) {
        throw new Exception("DB 오류 [좋아요 상태 확인 준비 실패]: " . $conn->error);
    }
    $stmt_check->bind_param("ii", $user_id, $review_id);
    $stmt_check->execute();
    $result_check = $stmt_check->get_result();
    $already_liked = $result_check->num_rows > 0;
    $stmt_check->close();

    if ($already_liked) {
        // 좋아요 취소: ReviewLikes 테이블에서 해당 행 삭제
        $sql_unlike = "DELETE FROM ReviewLikes WHERE user_id = ? AND review_id = ?";
        $stmt_unlike = $conn->prepare($sql_unlike);
        if (!$stmt_unlike) {
            throw new Exception("DB 오류 [좋아요 취소 준비 실패]: " . $conn->error);
        }
        $stmt_unlike->bind_param("ii", $user_id, $review_id);
        if (!$stmt_unlike->execute()) {
            throw new Exception("좋아요 취소 실행 실패: " . $stmt_unlike->error);
        }
        $stmt_unlike->close();
        $response['liked'] = false;
        // $response['message'] = '좋아요가 취소되었습니다.'; // 성공 메시지는 생략 가능
    } else {
        // 좋아요: ReviewLikes 테이블에 새 행 추가
        $sql_like = "INSERT INTO ReviewLikes (user_id, review_id, created_at) VALUES (?, ?, NOW())";
        $stmt_like = $conn->prepare($sql_like);
        if (!$stmt_like) {
            throw new Exception("DB 오류 [좋아요 추가 준비 실패]: " . $conn->error);
        }
        $stmt_like->bind_param("ii", $user_id, $review_id);
        if (!$stmt_like->execute()) {
            // PK 중복(user_id, review_id) 등으로 인한 실패 가능성도 고려할 수 있으나,
            // 이미 위에서 $already_liked로 확인했으므로 다른 DB 오류일 가능성
            throw new Exception("좋아요 추가 실행 실패: " . $stmt_like->error);
        }
        $stmt_like->close();
        $response['liked'] = true;
        // $response['message'] = '좋아요를 눌렀습니다.'; // 성공 메시지는 생략 가능
    }

    // 새로운 총 좋아요 수 계산
    $sql_count_likes = "SELECT COUNT(*) AS like_count FROM ReviewLikes WHERE review_id = ?";
    $stmt_count = $conn->prepare($sql_count_likes);
    if (!$stmt_count) {
        throw new Exception("DB 오류 [좋아요 수 집계 준비 실패]: " . $conn->error);
    }
    $stmt_count->bind_param("i", $review_id);
    $stmt_count->execute();
    $result_count = $stmt_count->get_result();
    $row_count = $result_count->fetch_assoc();
    $response['new_like_count'] = (int)$row_count['like_count'];
    $stmt_count->close();

    $conn->commit(); // 모든 작업 성공 시 커밋
    $response['success'] = true;

} catch (Exception $e) {
    $conn->rollback(); // 오류 발생 시 롤백
    $response['message'] = $e->getMessage();
    error_log("toggle_review_like.php: Transaction error - " . $e->getMessage());
}

echo json_encode($response);
$conn->close();
?>